%{
/*
Copyright (C) 2000-2013  The PARI group.

This file is part of the GP2C package.

PARI/GP is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation. It is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY WHATSOEVER.

Check the License for details. You should have received a copy of it, along
with the package; see the file 'COPYING'. If not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.*/

#define YYDEBUG 1
#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "header.h"
int yyerror(const char *s);
int yylex(void);
static int once=0;
static int newcommand(int x, int y)
{
  return (x==GNOARG)?y:(y==GNOARG)?x:newnode(Fseq,x,y);
}
%}
%define parse.error verbose
%union {
  int val;
  strcom str;
}
%token KPARROW ")->"
%token KARROW "->"
%token KDOTDOT ".."
%token KPE   "+="
%token KSE   "-="
%token KME   "*="
%token KDE   "/="
%token KDRE  "\\/="
%token KEUCE "\\="
%token KMODE "%="
%token KAND  "&&"
%token KOR   "||"
%token KID   "==="
%token KEQ   "=="
%token KNE   "!="
%token KGE   ">="
%token KLE   "<="
%token KSRE  ">>="
%token KSLE  "<<="
%token KSR   ">>"
%token KSL   "<<"
%token KDR   "\\/"
%token KPP   "++"
%token KSS   "--"
%token <str> KINTEGER "integer"
%token <str> KREAL "real number"
%token <str> KENTRY "variable name"
%token <str> KSTRING "character string"
%left '\n'
%left SEQ DEFFUNC
%left INT LVAL
%right ")->" "->"
%left ';' ','
%right '=' "+=" "-=" "*=" "/=" "\\/=" "\\=" "%=" ">>=" "<<="
%left '&' "&&" "||"
%left "===" "==" "!=" '>' ">=" '<' "<="
%left '+' '-'
%left '%' "\\/" '\\' '/' '*' ">>" "<<"
%left SIGN
%right '^'
%left '#'
%left '!' '~' '[' DERIV
%left '\''
%left '.' MAT
%left "++" "--"
%left '('
%left ':'
%type <val> commands seq
%type <val> range matrix matrix_index expr exprno
%type <val> lvalue deriv
%type <val> matrixelts matrixeltsno matrixlines arg listarg definition
%type <val> funcid memberid
%type <val> backticks history
%type <val> compr in inseq
%%

commands: seq               {$$=$1;}
        | commands '\n' seq {$$=newcommand($1,$3);}
;

seq: /**/ %prec SEQ  {$$=GNOARG;}
   | expr %prec SEQ  {$$=$1;}
   | seq ';'         {$$=$1;}
   | seq ';' expr    {$$=newcommand($1,$3);}
;

range: /* */          { $$=newnode(Frange,GNORANGE,GNORANGE); }
     | expr           { $$=newnode(Frange,$1,GNORANGE); }
     | expr ".." expr { $$=newnode(Frange,$1,$3); }
     | '^' expr       { $$=newnode(Frange,GNORANGE,$2); }
;

matrix_index: '[' range ',' range ']' {$$=newnode(Fmatrix,$2,$4);}
            | '[' range ']'           {$$=newnode(Fmatrix,$2,-1);}
;

backticks: '`' {$$=1;}
         | backticks '`' {$$=$1+1;}
;

history: '%'           {$$=newopcall(OPhist,-1,-1);}
       | '%' KINTEGER  {$$=newopcall(OPhist,newintnode($2.s,$2.c),-1);}
       | '%' backticks {$$=newopcall(OPhist,newnode(Fsmall,-$2,-1),-1);}
       | '%' '#'          {$$=newopcall(OPhisttime,-1,-1);}
       | '%' '#' KINTEGER {$$=newopcall(OPhisttime,newintnode($3.s,$3.c),-1);}
       | '%' '#' backticks{$$=newopcall(OPhisttime,newnode(Fsmall,-$3,-1),-1);}
;

deriv: '\'' {$$ = 1;}
     | deriv '\'' {$$ = $1+1;}
;

expr: KINTEGER %prec INT  {$$=newintnode($1.s,$1.c);}
    | KREAL               {$$=newrealnode($1.s,$1.c);}
    | '.'                 {$$=newnode(Fconst,newsmallrealvalue(0),-1);}
    | KINTEGER '.' KENTRY {$$=newnodecom(Ffunction,newmember($3.s),newintnode($1.s,$1.c),
                                                   $3.c);}
    | KSTRING       {$$=newstringnode($1.s,$1.c);}
    | '\'' KENTRY   {$$=newquotenode($2.s,$2.c);}
    | history           {$$=$1;}
    | expr '(' listarg ')'  {$$=newnode(Fcall,$1,$3);}
    | funcid            {$$=$1;}
    | lvalue %prec LVAL {$$=$1;}
    | matrix            {$$=$1;}
    | compr             {$$=$1;}
    | definition        {$$=$1;}
    | matrix '=' expr {$$=newnode(Fassign,$1,$3);}
    | lvalue '=' expr {$$=newnode(Fassign,$1,$3);}
    | lvalue "++"     {$$=newopcall(OPpp,$1,-1);}
    | lvalue "--"     {$$=newopcall(OPss,$1,-1);}
    | lvalue "*="   expr {$$=newopcall(OPme,$1,$3);}
    | lvalue "/="   expr {$$=newopcall(OPde,$1,$3);}
    | lvalue "\\/=" expr {$$=newopcall(OPdre,$1,$3);}
    | lvalue "\\="  expr {$$=newopcall(OPeuce,$1,$3);}
    | lvalue "%="   expr {$$=newopcall(OPmode,$1,$3);}
    | lvalue "<<="  expr {$$=newopcall(OPsle,$1,$3);}
    | lvalue ">>="  expr {$$=newopcall(OPsre,$1,$3);}
    | lvalue "+="   expr {$$=newopcall(OPpe,$1,$3);}
    | lvalue "-="   expr {$$=newopcall(OPse,$1,$3);}
    | '!' expr         {$$=newopcall(OPnb,$2,-1);}
    | '#' expr         {$$=newopcall(OPlength,$2,-1);}
    | expr "||"  expr  {$$=newopcall(OPor,$1,$3);}
    | expr "&&"  expr  {$$=newopcall(OPand,$1,$3);}
    | expr '&'   expr  {$$=newopcall(OPand,$1,$3);}
    | expr "===" expr  {$$=newopcall(OPid,$1,$3);}
    | expr "=="  expr  {$$=newopcall(OPeq,$1,$3);}
    | expr "!="  expr  {$$=newopcall(OPne,$1,$3);}
    | expr ">="  expr  {$$=newopcall(OPge,$1,$3);}
    | expr '>'   expr  {$$=newopcall(OPg,$1,$3);}
    | expr "<="  expr  {$$=newopcall(OPle,$1,$3);}
    | expr '<'   expr  {$$=newopcall(OPl,$1,$3);}
    | expr '-'   expr  {$$=newopcall(OPs,$1,$3);}
    | expr '+'   expr  {$$=newopcall(OPp,$1,$3);}
    | expr "<<"  expr  {$$=newopcall(OPsl,$1,$3);}
    | expr ">>"  expr  {$$=newopcall(OPsr,$1,$3);}
    | expr '%'   expr  {$$=newopcall(OPmod,$1,$3);}
    | expr "\\/" expr  {$$=newopcall(OPdr,$1,$3);}
    | expr '\\'  expr  {$$=newopcall(OPeuc,$1,$3);}
    | expr '/'   expr  {$$=newopcall(OPd,$1,$3);}
    | expr '*'   expr  {$$=newopcall(OPm,$1,$3);}
    | '+' expr %prec SIGN {$$=newopcall(OPpl,$2,-1);}
    | '-' expr %prec SIGN {$$=newopcall(OPn,$2,-1);}
    | expr '^' expr {$$=newopcall(OPpow,$1,$3);}
    | expr '~' {$$=newopcall(OPtrans,$1,-1);}
    | expr deriv %prec DERIV { $$=newderivn($1,$2); }
    | expr '!'  {$$=newopcall(OPfact,$1,-1);}
    | expr '#'  {$$=newopcall(OPprim,$1,-1);}
    | expr matrix_index %prec MAT {$$=newnode(Fmatcoeff,$1,$2);}
    | memberid {$$=$1;}
    | expr ':' KENTRY   {$$=newtag($1,$3.s,$3.c);}
    | '(' expr ')' {$$=$2;}
;

lvalue: KENTRY %prec LVAL   {$$=newnodecom(Fentry,newentry($1.s),-1,$1.c);}
      | lvalue matrix_index {$$=newnode(Fmatcoeff,$1,$2);}
      | lvalue ':' KENTRY   {$$=newtag($1,$3.s,$3.c);}
;


exprno: expr {$$=$1;}
      | /**/ {$$=GNOARG;}
;

matrixeltsno: matrixelts {$$=$1;}
            | /**/ {$$=GNOARG;}
;

matrixelts: expr {$$=$1;}
          | matrixeltsno ',' exprno {$$=newnode(Fmatrixelts,$1,$3);}
;

matrixlines: matrixelts  ';' matrixelts {$$=newnode(Fmatrixlines,$1,$3);}
           | matrixlines ';' matrixelts {$$=newnode(Fmatrixlines,$1,$3);}
;

matrix: '[' ']'             {$$=newnode(Fvec,-1,-1);}
      | '[' expr ".." expr ']' {$$=newopcall(OPrange,$2,$4);}
      | '[' ';' ']'         {$$=newnode(Fmat,-1,-1);}
      | '[' matrixelts ']'  {$$=newnode(Fvec,$2,-1);}
      | '[' matrixlines ']' {$$=newnode(Fmat,$2,-1);}
      | '[' error ']'       {YYABORT;}
;

in: lvalue '<' '-' expr {$$=newnode(Flistarg,$4,$1);}
;

inseq: in                    {$$=newopcall(OPcompr,$1,-2);}
     | in ',' expr           {$$=newopcall3(OPcompr,$1,-2,$3);}
     | in ';' inseq          {$$=newopcall(OPcomprc,$1,$3);}
     | in ',' expr ';' inseq {$$=newopcall3(OPcomprc,$1,$5,$3);}
;

compr: '[' expr '|' inseq ']' {$$=addcurrexpr($4,$2);}
;

arg: seq        {$$=$1;}
   | lvalue '[' ".." ']' {$$=newnode(Fvararg,$1,-1);}
   | '&' lvalue {$$=newnode(Frefarg,$2,-1);}
   | '~' lvalue {$$=newnode(Findarg,$2,-1);}
   | arg error  {if (!once) { yyerrok; s_errors.n--;} once=1;}  expr
                     {once=0; $$=newopcall(OPcat,$1,$4);}
;

listarg: arg {$$=$1;}
       | listarg ',' arg {$$=newnode(Flistarg,$1,$3);}
;

funcid: KENTRY '(' listarg ')' {$$=newnodecom(Ffunction,newentry($1.s),$3,$1.c);}
;

memberid: expr '.' KENTRY {$$=newnodecom(Ffunction,newmember($3.s),$1,$3.c);}
;

definition: funcid   '=' seq %prec DEFFUNC
                                   {$$=newnode(Fdeffunc,$1,$3);}
          | memberid '=' seq %prec DEFFUNC
                                   {$$=newnode(Fdeffunc,$1,$3);}
          | lvalue "->" seq              {$$=newnode(Flambda, $1,$3);}
          | '(' listarg ")->" seq        {$$=newnode(Flambda, $2,$4);}
;

%%
int yyerror(const char *s)
{
  int n=stack_new(&s_errors);
  errors[n].fileno=filecount;
  errors[n].lineno=linecount;
  errors[n].txt=strdup(s);
  if (yydebug) fprintf(stderr,"%s:%d: %s\n",srcfile[filecount],linecount,s);
  return 0;
}
